# BotAJS Git Integration - Pure PHP

A pure PHP backend solution for BotAJS that enables SSH-based Git export/import of bot commands to/from GitHub.

## Features

- Generate SSH deploy keys for secure GitHub access
- Export bot commands from BotAJS to GitHub repository
- Import bot commands from GitHub back to BotAJS
- Session-based key management
- RESTful API endpoints
- Pure PHP implementation with Composer dependencies

## Requirements

- PHP >= 7.4
- Composer
- Git CLI
- SSH support

## Installation

1. Clone or download this repository:

```bash
cd /path/to/php-git-bota
```

2. Install Composer dependencies:

```bash
composer install
```

3. Configure environment variables:

```bash
cp .env.example .env
```

Edit `.env` and set your API key:

```env
API_KEY=your-secret-api-key-here
GIT_STORAGE_PATH=/tmp/botajs_git
PHP_SESSION_PATH=/tmp/botajs_sessions
```

4. Ensure storage directories exist with proper permissions:

```bash
chmod 755 storage/git storage/sessions
```

## Project Structure

```
project/
├── src/                      # Core PHP classes
│   ├── BotAJSClient.php     # BotAJS API client
│   ├── GitHandler.php       # Git operations handler
│   ├── SessionManager.php   # Session management
│   ├── Request.php          # HTTP request handler
│   ├── Response.php         # HTTP response helper
│   ├── Auth.php             # Authentication
│   └── Config.php           # Configuration loader
├── api/                      # API endpoints
│   ├── generate-key.php     # Generate SSH keys
│   ├── get-key.php          # Retrieve public key
│   ├── export-bot.php       # Export bot to GitHub
│   └── import-bot.php       # Import bot from GitHub
├── public/
│   └── index.php            # Main entry point/router
├── storage/                  # Storage directories
│   ├── git/                 # Temporary git operations
│   └── sessions/            # PHP sessions
├── composer.json            # Dependencies
├── .env                     # Environment config
└── README.md
```

## API Endpoints

### 1. Generate SSH Deploy Key

**POST** `/api/git/generate-key`

Generate a new SSH key pair for GitHub authentication.

**Headers:**
```
X-API-KEY: your-secret-api-key-here
Content-Type: application/json
```

**Request Body:**
```json
{
  "user_id": "user123"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Deploy key generated successfully",
  "public_key": "ssh-rsa AAAAB3NzaC1yc2EA...",
  "note": "Copy this key and add it as a Deploy Key in your GitHub repository settings"
}
```

**Setup Instructions:**
1. Copy the returned `public_key`
2. Go to your GitHub repository → Settings → Deploy keys
3. Click "Add deploy key"
4. Paste the public key
5. Check "Allow write access"
6. Save

---

### 2. Get Public Key

**POST** `/api/git/get-key`

Retrieve the previously generated public SSH key.

**Headers:**
```
X-API-KEY: your-secret-api-key-here
Content-Type: application/json
```

**Request Body:**
```json
{
  "user_id": "user123"
}
```

**Response:**
```json
{
  "public_key": "ssh-rsa AAAAB3NzaC1yc2EA..."
}
```

---

### 3. Export Bot to GitHub

**POST** `/api/bots/export`

Export a bot and all its commands from BotAJS to a GitHub repository.

**Headers:**
```
X-API-KEY: your-secret-api-key-here
Content-Type: application/json
```

**Request Body:**
```json
{
  "user_id": "user123",
  "botId": "bot_123",
  "git_repo_url": "git@github.com:username/repo.git",
  "git_branch": "main",
  "commit_message": "Update bot commands",
  "user_email": "user@example.com",
  "api_key": "botajs_api_key_here"
}
```

**Parameters:**
- `user_id` - User identifier (for session management)
- `botId` - BotAJS bot ID to export
- `git_repo_url` - SSH Git repository URL
- `git_branch` - Target branch (default: main)
- `commit_message` - Optional commit message
- `user_email` - Git commit author email
- `api_key` - BotAJS API key

**Response:**
```json
{
  "success": true,
  "message": "Bot exported successfully",
  "commands_exported": 5
}
```

**Generated Structure:**
```
repo/
└── bots/
    └── bot_123/
        ├── bot.json          # Bot metadata
        ├── README.md         # Bot documentation
        └── commands/
            ├── start.js
            ├── help.js
            └── default.js    # Default handler (*)
```

---

### 4. Import Bot from GitHub

**POST** `/api/bots/import`

Import bot commands from a GitHub repository back to BotAJS.

**Headers:**
```
X-API-KEY: your-secret-api-key-here
Content-Type: application/json
```

**Request Body:**
```json
{
  "user_id": "user123",
  "botId": "bot_123",
  "git_repo_url": "git@github.com:username/repo.git",
  "git_branch": "main",
  "api_key": "botajs_api_key_here"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Bot imported successfully",
  "commands_imported": 5
}
```

---

## Usage Example with cURL

### 1. Generate SSH Key

```bash
curl -X POST http://localhost/api/git/generate-key \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{"user_id": "user123"}'
```

### 2. Export Bot to GitHub

```bash
curl -X POST http://localhost/api/bots/export \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user123",
    "botId": "bot_123",
    "git_repo_url": "git@github.com:username/repo.git",
    "git_branch": "main",
    "commit_message": "Update bot commands",
    "user_email": "user@example.com",
    "api_key": "botajs_api_key_here"
  }'
```

### 3. Import Bot from GitHub

```bash
curl -X POST http://localhost/api/bots/import \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user123",
    "botId": "bot_123",
    "git_repo_url": "git@github.com:username/repo.git",
    "git_branch": "main",
    "api_key": "botajs_api_key_here"
  }'
```

---

## Web Server Configuration

### Apache (.htaccess in public/)

```apache
RewriteEngine On
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^ index.php [QSA,L]
```

### Nginx

```nginx
server {
    listen 80;
    server_name your-domain.com;
    root /path/to/php-git-bota/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }
}
```

---

## Security Considerations

1. **API Key Protection**: Never commit your `.env` file to version control
2. **SSH Keys**: Private keys are stored in PHP sessions and temporary directories
3. **Cleanup**: Temporary directories are automatically cleaned after operations
4. **Authentication**: All endpoints require valid API key
5. **SSH Host Verification**: Disabled for automation (use with caution in production)

---

## Development

### Running with PHP Built-in Server

```bash
cd public
php -S localhost:8000
```

Access endpoints at: `http://localhost:8000/api/...`

---

## Troubleshooting

### SSH Connection Issues

- Ensure Git is installed: `git --version`
- Check SSH key permissions: Private keys must be `0600`
- Verify GitHub deploy key has write access

### Permission Errors

```bash
chmod -R 755 storage/
chmod -R 777 /tmp/botajs_git /tmp/botajs_sessions
```

### Session Issues

- Check PHP session directory permissions
- Verify `session.save_path` in `php.ini`
- Clear sessions: `rm -rf storage/sessions/*`

---

## License

MIT License

---

## Contributing

Contributions are welcome! Please submit a pull request or open an issue.

---

## Support

For issues related to:
- BotAJS API: https://botajs.top
- This integration: Open a GitHub issue

---

## Changelog

### v1.0.0
- Initial release
- SSH key generation
- Export/import functionality
- Session-based key management
- RESTful API endpoints
