# Database Setup Guide

## Overview

The BotAJS Git Integration now uses MySQL database for persistent SSH key storage. This replaces the previous session-only storage, making keys persistent across server restarts and accessible from cPanel environments.

## SSH Key Architecture

**One SSH key per user_id** (NOT per bot)

- SSH keys authenticate with GitHub on behalf of a **user**, not individual bots
- One user can manage multiple bots using the same SSH key
- GitHub deploy keys are per-repository, not per-bot
- Keys are stored by `user_id` in the database

## Database Configuration

### Database Credentials

```
Database Name: dannvtuc_botajs
Username: dannvtuc_botajs
Password: [Your cPanel password]
Host: localhost
```

### Step 1: Create the Database Table

Run the following SQL in your cPanel phpMyAdmin or MySQL command line:

```sql
-- SSH Keys Table
-- Stores SSH key pairs for users
-- One key pair per user (can be used for multiple bots)
CREATE TABLE IF NOT EXISTS `ssh_keys` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id` VARCHAR(255) NOT NULL,
  `public_key` TEXT NOT NULL,
  `private_key` TEXT NOT NULL,
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_user_id` (`user_id`),
  KEY `idx_user_id` (`user_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

### Step 2: Update .env Configuration

Update your [.env](.env) file with database credentials:

```env
# Database Configuration
DB_HOST=localhost
DB_NAME=dannvtuc_botajs
DB_USER=dannvtuc_botajs
DB_PASSWORD=your_cpanel_password_here
DB_CHARSET=utf8mb4
```

### Step 3: Import SQL Schema (Alternative)

You can also import the schema file directly:

```bash
# Using MySQL command line
mysql -u dannvtuc_botajs -p dannvtuc_botajs < database/schema.sql

# Or upload database/schema.sql via cPanel phpMyAdmin
```

## API Endpoints

### 1. Generate SSH Key
**Endpoint:** `POST /api/git/generate-key`

Generates a new SSH key pair and stores it in the database.

```bash
curl -X POST http://your-domain.com/api/git/generate-key \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{"user_id": "user_123"}'
```

**Response:**
```json
{
  "success": true,
  "message": "Deploy key generated successfully",
  "data": {
    "public_key": "ssh-rsa AAAAB3NzaC1yc2EA...",
    "note": "Copy this key and add it as a Deploy Key in your GitHub repository settings"
  }
}
```

### 2. Get User SSH Keys
**Endpoint:** `POST /api/user/ssh-keys`

Retrieves SSH key information for a user (for display in your app).

```bash
curl -X POST http://your-domain.com/api/user/ssh-keys \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{"user_id": "user_123"}'
```

**Response (Key exists):**
```json
{
  "has_key": true,
  "message": "SSH key found",
  "public_key": "ssh-rsa AAAAB3NzaC1yc2EA...",
  "created_at": "2025-11-21 10:30:00",
  "updated_at": "2025-11-21 10:30:00",
  "key_fingerprint": "2f:3a:4b:5c:6d:7e:8f:9a:0b:1c:2d:3e:4f:5a:6b:7c"
}
```

**Response (No key):**
```json
{
  "has_key": false,
  "message": "No SSH key configured for this user",
  "public_key": null,
  "created_at": null,
  "updated_at": null
}
```

### 3. Export Bot to GitHub
**Endpoint:** `POST /api/bots/export`

Now automatically retrieves SSH key from database.

```bash
curl -X POST http://your-domain.com/api/bots/export \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user_123",
    "botId": "YOUR_BOT_ID",
    "git_repo_url": "git@github.com:username/repo.git",
    "git_branch": "main",
    "commit_message": "Update bot commands",
    "user_email": "user@example.com",
    "api_key": "YOUR_BOTAJS_API_KEY"
  }'
```

### 4. Import Bot from GitHub
**Endpoint:** `POST /api/bots/import`

Now automatically retrieves SSH key from database.

```bash
curl -X POST http://your-domain.com/api/bots/import \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user_123",
    "botId": "YOUR_BOT_ID",
    "git_repo_url": "git@github.com:username/repo.git",
    "git_branch": "main",
    "api_key": "YOUR_BOTAJS_API_KEY"
  }'
```

## Database Class Features

The new [Database.php](src/Database.php) class provides:

```php
// Get singleton instance
$db = Database::getInstance();

// Store or update SSH key
$db->storeSSHKey($userId, $publicKey, $privateKey);

// Get SSH key data
$keyData = $db->getSSHKey($userId);
// Returns: ['user_id', 'public_key', 'private_key', 'created_at', 'updated_at']

// Check if user has SSH key
$hasKey = $db->hasSSHKey($userId);
// Returns: true/false

// Delete SSH key
$deleted = $db->deleteSSHKey($userId);
// Returns: true/false
```

## Migration from Session Storage

The system maintains **backward compatibility** with session storage:

1. **Priority:** Database is checked first
2. **Fallback:** If not in database, checks session storage
3. **Dual Storage:** New keys are stored in both database AND session

This ensures:
- Existing sessions continue to work
- Gradual migration to database storage
- No disruption to active users

## Security Notes

1. **Private keys** are stored encrypted in the database
2. The `/api/user/ssh-keys` endpoint **never returns private keys** (only public keys)
3. Private keys are only accessed internally for Git operations
4. Use strong database passwords in production
5. Ensure `.env` file is not accessible via web

## cPanel Deployment

### Upload Files
1. Upload all files to your cPanel hosting
2. Set document root to `/public` directory
3. Ensure proper permissions: `chmod 755` for directories

### Configure Database
1. Go to cPanel → MySQL Databases
2. Create database: `dannvtuc_botajs` (if not exists)
3. Create user: `dannvtuc_botajs` (if not exists)
4. Assign all privileges to the user
5. Import `database/schema.sql` via phpMyAdmin

### Update .env
1. Edit `.env` file with your actual database password
2. Update API_KEY with a secure random key
3. Adjust paths for cPanel environment if needed

### Test Connection
Run `php test-setup.php` to verify database connectivity.

## Troubleshooting

### Connection Failed
```
Error: Database connection failed: Access denied
```
**Solution:** Check database credentials in `.env`

### Table Not Found
```
Error: Table 'dannvtuc_botajs.ssh_keys' doesn't exist
```
**Solution:** Import `database/schema.sql`

### Permission Denied
```
Error: Access denied for user 'dannvtuc_botajs'@'localhost'
```
**Solution:** Grant privileges in cPanel MySQL Databases

## Example App Integration

Display user SSH key status in your app:

```javascript
// Fetch user SSH key status
async function getUserSSHKey(userId) {
  const response = await fetch('/api/user/ssh-keys', {
    method: 'POST',
    headers: {
      'X-API-KEY': 'your-api-key',
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({ user_id: userId })
  });

  const data = await response.json();

  if (data.has_key) {
    console.log('SSH Key configured');
    console.log('Fingerprint:', data.key_fingerprint);
    console.log('Created:', data.created_at);
    // Display in UI: "✓ SSH Key Configured"
  } else {
    console.log('No SSH key configured');
    // Display in UI: "Generate SSH Key" button
  }
}
```

## Schema Changes

If you need to modify the schema in the future:

1. Always backup the database first
2. Use `ALTER TABLE` for changes
3. Test on development environment
4. Document changes in migration files

## Support

For issues:
- Database connectivity: Check cPanel MySQL settings
- SSH key generation: Check `test-setup.php` output
- API errors: Check PHP error logs in cPanel
