# Database Storage Implementation - Changes Summary

## What Changed

### SSH Key Storage Architecture

**Previous:** SSH keys stored in PHP sessions only (temporary, lost on restart)
**New:** SSH keys stored in MySQL database (persistent, survives restarts)

### One SSH Key Per User (Not Per Bot)

**Architecture Decision:**
- ✓ One SSH key per `user_id`
- ✗ NOT one SSH key per `botId`

**Rationale:**
- SSH keys authenticate with GitHub on behalf of a **user**, not individual bots
- One user can manage multiple bots using the same SSH key
- GitHub deploy keys are per-repository, not per-bot
- Reduces key management complexity

**Example:**
```
user_id: "john_123"
  └── SSH Key (stored once)
      ├── Bot A (uses this key)
      ├── Bot B (uses this key)
      └── Bot C (uses this key)
```

## New Files

### Core Files
1. **src/Database.php** - MySQL database connection and SSH key management
2. **database/schema.sql** - SQL schema for `ssh_keys` table
3. **api/user-ssh-keys.php** - New API endpoint to fetch user SSH keys

### Documentation
4. **DATABASE_SETUP.md** - Complete database setup guide
5. **migrate-database.php** - Automated migration script
6. **CHANGES.md** - This file

## Modified Files

### Configuration
- **.env** - Added database credentials (DB_HOST, DB_NAME, DB_USER, DB_PASSWORD)

### API Endpoints (Updated to use database)
- **api/generate-key.php** - Now stores keys in database + session
- **api/get-key.php** - Now retrieves from database (fallback to session)
- **api/export-bot.php** - Now retrieves key from database (fallback to session)
- **api/import-bot.php** - Now retrieves key from database (fallback to session)

### Router
- **public/index.php** - Added route for `/api/user/ssh-keys`

## New API Endpoint

### POST /api/user/ssh-keys

**Purpose:** Retrieve SSH key information for display in your app

**Request:**
```json
{
  "user_id": "john_123"
}
```

**Response (Has Key):**
```json
{
  "has_key": true,
  "message": "SSH key found",
  "public_key": "ssh-rsa AAAAB3NzaC1yc2EA...",
  "created_at": "2025-11-21 10:30:00",
  "updated_at": "2025-11-21 10:30:00",
  "key_fingerprint": "2f:3a:4b:5c:6d:7e:8f:9a:0b:1c:2d:3e:4f:5a:6b:7c"
}
```

**Response (No Key):**
```json
{
  "has_key": false,
  "message": "No SSH key configured for this user",
  "public_key": null,
  "created_at": null,
  "updated_at": null
}
```

**Security:** Never returns private keys (only public keys)

## Database Schema

### Table: ssh_keys

```sql
CREATE TABLE `ssh_keys` (
  `id` INT(11) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
  `user_id` VARCHAR(255) NOT NULL UNIQUE,
  `public_key` TEXT NOT NULL,
  `private_key` TEXT NOT NULL,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  KEY `idx_user_id` (`user_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
```

**Features:**
- `user_id` is UNIQUE (one key per user)
- Automatic timestamps for created_at/updated_at
- Index on user_id for fast lookups

## Backward Compatibility

**Dual Storage Strategy:**
- New keys are stored in BOTH database AND session
- Reading checks database FIRST, falls back to session
- Ensures existing sessions continue to work
- Gradual migration without disruption

**Migration Path:**
1. User with session-only key → continues to work
2. User generates new key → stored in database
3. Over time, all users migrate to database storage

## Setup Instructions

### Quick Setup (3 steps)

```bash
# 1. Update .env with database password
nano .env
# Add: DB_PASSWORD=your_cpanel_password

# 2. Run migration script
php migrate-database.php

# 3. Verify installation
php test-setup.php
```

### cPanel Setup

See [DATABASE_SETUP.md](DATABASE_SETUP.md) for detailed cPanel instructions.

## App Integration Example

```javascript
// Check if user has SSH key configured
async function checkUserSSHKey(userId) {
  const response = await fetch('/api/user/ssh-keys', {
    method: 'POST',
    headers: {
      'X-API-KEY': 'your-api-key',
      'Content-Type': 'application/json'
    },
    body: JSON.stringify({ user_id: userId })
  });

  const data = await response.json();

  if (data.has_key) {
    // Show: "✓ SSH Key Configured"
    // Display: Fingerprint, Created date
    // Enable: Export/Import buttons
  } else {
    // Show: "⚠️ No SSH Key"
    // Display: "Generate SSH Key" button
    // Disable: Export/Import buttons
  }
}
```

## Benefits

### For cPanel Deployment
✓ Keys persist across server restarts
✓ Keys survive PHP session expiration
✓ Database backup = key backup
✓ Easy key management via phpMyAdmin

### For Users
✓ Generate key once, use for all bots
✓ Check key status via API
✓ No need to regenerate after session timeout
✓ Consistent experience across devices

### For Developers
✓ Clean API for key management
✓ Singleton pattern for database connection
✓ Prepared statements (SQL injection safe)
✓ Easy to add new features

## Testing

### Test Database Connection
```bash
php migrate-database.php
```

### Test SSH Key Generation
```bash
curl -X POST http://localhost:8000/api/git/generate-key \
  -H "X-API-KEY: your-api-key" \
  -H "Content-Type: application/json" \
  -d '{"user_id": "test_user"}'
```

### Test SSH Key Retrieval
```bash
curl -X POST http://localhost:8000/api/user/ssh-keys \
  -H "X-API-KEY: your-api-key" \
  -H "Content-Type: application/json" \
  -d '{"user_id": "test_user"}'
```

### Test Export (with database key)
```bash
curl -X POST http://localhost:8000/api/bots/export \
  -H "X-API-KEY: your-api-key" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "test_user",
    "botId": "YOUR_BOT_ID",
    "git_repo_url": "git@github.com:username/repo.git",
    "git_branch": "main",
    "commit_message": "Test export with database",
    "user_email": "test@example.com",
    "api_key": "YOUR_BOTAJS_API_KEY"
  }'
```

## Security Notes

1. **Private keys** stored in database (ensure database is secure)
2. **Public keys only** returned via `/api/user/ssh-keys` endpoint
3. **Strong passwords** required for database in production
4. **`.env` file** must not be web-accessible (already in .gitignore)
5. **Prepared statements** used throughout (SQL injection safe)

## Troubleshooting

### Database Connection Failed
```
Error: Database connection failed: Access denied
```
**Fix:** Check `DB_PASSWORD` in `.env`

### Table Not Found
```
Error: Table 'dannvtuc_botajs.ssh_keys' doesn't exist
```
**Fix:** Run `php migrate-database.php`

### SSH Key Not Found
```
Error: SSH key not configured
```
**Fix:** Generate key: `POST /api/git/generate-key`

## Next Steps

1. ✓ Database implementation complete
2. ✓ New API endpoint created
3. ✓ Migration script ready
4. ⏳ Update your app UI to show SSH key status
5. ⏳ Deploy to cPanel
6. ⏳ Test with real bot export/import

## Support

- Full documentation: [DATABASE_SETUP.md](DATABASE_SETUP.md)
- Migration script: `php migrate-database.php`
- Test script: `php test-setup.php`
- Main README: [README.md](README.md)

---

**Implementation Date:** 2025-11-21
**Version:** 2.0.0 (Database Storage)
