# API Documentation

Complete API reference for BotAJS Git Integration endpoints.

---

## Table of Contents

1. [Authentication](#authentication)
2. [SSH Key Management](#ssh-key-management)
   - [Generate SSH Key](#1-generate-ssh-key)
   - [Get Public Key](#2-get-public-key)
   - [Get User SSH Keys](#3-get-user-ssh-keys)
3. [Bot Operations](#bot-operations)
   - [Export Bot to GitHub](#4-export-bot-to-github)
   - [Import Bot from GitHub](#5-import-bot-from-github)
4. [Error Codes](#error-codes)
5. [Rate Limiting](#rate-limiting)
6. [Examples](#examples)

---

## Authentication

All API endpoints require authentication using an API key.

**Header Required:**
```
X-API-KEY: your-secret-api-key-here
```

**Content Type:**
```
Content-Type: application/json
```

### Authentication Errors

**401 Unauthorized**
```json
{
  "success": false,
  "message": "Unauthorized",
  "error": "Invalid or missing API key"
}
```

---

## SSH Key Management

### 1. Generate SSH Key

Generate a new SSH key pair for a user. This key will be used for Git operations.

**Endpoint:** `POST /api/git/generate-key`

**Architecture:** One SSH key per `user_id` (not per bot). One user can manage multiple bots with the same SSH key.

#### Request

```bash
curl -X POST http://your-domain.com/api/git/generate-key \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user_123"
  }'
```

**Parameters:**

| Parameter | Type   | Required | Description                                      |
|-----------|--------|----------|--------------------------------------------------|
| user_id   | string | No       | Unique user identifier. Defaults to "anonymous"  |

#### Responses

**✅ Success (200 OK)**
```json
{
  "success": true,
  "message": "Deploy key generated successfully",
  "data": {
    "public_key": "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAACAQCyk56tnn3vL8jKqZ...",
    "note": "Copy this key and add it as a Deploy Key in your GitHub repository settings"
  }
}
```

**❌ Error (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "SSH key generation failed",
  "error": "Failed to generate SSH key pair"
}
```

**❌ Database Error (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Database connection failed: Access denied for user 'dannvtuc_botajs'@'localhost'",
  "error": "Database connection failed: Access denied for user 'dannvtuc_botajs'@'localhost'"
}
```

#### Notes

- Generates a 4096-bit RSA key pair
- Public and private keys are stored in database
- Also stored in session for backward compatibility
- If user already has a key, it will be replaced
- Public key must be added to GitHub as a Deploy Key with write access

---

### 2. Get Public Key

Retrieve the public SSH key for a user.

**Endpoint:** `POST /api/git/get-key`

#### Request

```bash
curl -X POST http://your-domain.com/api/git/get-key \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user_123"
  }'
```

**Parameters:**

| Parameter | Type   | Required | Description                                      |
|-----------|--------|----------|--------------------------------------------------|
| user_id   | string | No       | Unique user identifier. Defaults to "anonymous"  |

#### Responses

**✅ Success (200 OK)**
```json
{
  "public_key": "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAACAQCyk56tnn3vL8jKqZ..."
}
```

**❌ Key Not Found (404 Not Found)**
```json
{
  "success": false,
  "message": "No SSH key generated yet",
  "error": "No SSH key generated yet"
}
```

**❌ Error (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Database error",
  "error": "Connection timeout"
}
```

#### Notes

- Retrieves from database first, falls back to session
- Only returns public key (private key never exposed via API)
- Returns 404 if no key exists for the user

---

### 3. Get User SSH Keys

Retrieve detailed SSH key information for a user (for display in your app).

**Endpoint:** `POST /api/user/ssh-keys`

**New in v2.0:** This endpoint is specifically designed for app integration to display key status.

#### Request

```bash
curl -X POST http://your-domain.com/api/user/ssh-keys \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user_123"
  }'
```

**Parameters:**

| Parameter | Type   | Required | Description                                      |
|-----------|--------|----------|--------------------------------------------------|
| user_id   | string | No       | Unique user identifier. Defaults to "anonymous"  |

#### Responses

**✅ Success - Key Exists (200 OK)**
```json
{
  "has_key": true,
  "message": "SSH key found",
  "public_key": "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAACAQCyk56tnn3vL8jKqZ...",
  "created_at": "2025-11-21 10:30:00",
  "updated_at": "2025-11-21 10:30:00",
  "key_fingerprint": "2f:3a:4b:5c:6d:7e:8f:9a:0b:1c:2d:3e:4f:5a:6b:7c"
}
```

**✅ Success - No Key (200 OK)**
```json
{
  "has_key": false,
  "message": "No SSH key configured for this user",
  "public_key": null,
  "created_at": null,
  "updated_at": null
}
```

**❌ Error (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Database connection failed",
  "error": "Connection timeout"
}
```

#### Response Fields

| Field           | Type    | Description                                           |
|-----------------|---------|-------------------------------------------------------|
| has_key         | boolean | Whether user has an SSH key configured                |
| message         | string  | Human-readable status message                         |
| public_key      | string  | SSH public key (null if no key)                       |
| created_at      | string  | Timestamp when key was created (null if no key)       |
| updated_at      | string  | Timestamp when key was last updated (null if no key)  |
| key_fingerprint | string  | MD5 fingerprint of the key (null if no key)           |

#### Notes

- **Never returns private keys** (security feature)
- Returns 200 even when no key exists (check `has_key` field)
- Use this endpoint to show key status in your app UI
- Fingerprint format: `2f:3a:4b:5c:6d:...` (MD5 hash)

---

## Bot Operations

### 4. Export Bot to GitHub

Export bot commands and configuration from BotAJS to a GitHub repository.

**Endpoint:** `POST /api/bots/export`

#### Request

```bash
curl -X POST http://your-domain.com/api/bots/export \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user_123",
    "botId": "bot_abc123",
    "git_repo_url": "git@github.com:username/my-bot-repo.git",
    "git_branch": "main",
    "commit_message": "Update bot commands",
    "user_email": "user@example.com",
    "api_key": "botajs_api_key_here"
  }'
```

**Parameters:**

| Parameter      | Type   | Required | Default              | Description                                    |
|----------------|--------|----------|----------------------|------------------------------------------------|
| user_id        | string | No       | "anonymous"          | User identifier (must have SSH key configured) |
| botId          | string | **Yes**  | -                    | BotAJS bot identifier                          |
| git_repo_url   | string | **Yes**  | -                    | Git repository SSH URL                         |
| git_branch     | string | No       | "main"               | Target branch for commit                       |
| commit_message | string | No       | "Update bot config"  | Git commit message                             |
| user_email     | string | No       | "bot@botajs.top"     | Git commit author email                        |
| api_key        | string | **Yes**  | -                    | BotAJS API key                                 |

#### Responses

**✅ Success (200 OK)**
```json
{
  "success": true,
  "message": "Bot exported successfully to GitHub",
  "data": {
    "message": "Bot exported successfully to GitHub",
    "bot_id": "bot_abc123",
    "repository": "git@github.com:username/my-bot-repo.git",
    "branch": "main",
    "commit": "a3f2b1c",
    "files_exported": 5,
    "commands_exported": 3
  }
}
```

**❌ Validation Error (400 Bad Request)**
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": {
    "botId": "The botId field is required",
    "git_repo_url": "The git_repo_url field is required",
    "api_key": "The api_key field is required"
  }
}
```

**❌ SSH Key Not Configured (400 Bad Request)**
```json
{
  "success": false,
  "message": "SSH key not configured. Generate one first.",
  "error": "SSH key not configured. Generate one first."
}
```

**❌ Bot Not Found (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Bot with ID 'bot_abc123' not found",
  "error": "Bot with ID 'bot_abc123' not found"
}
```

**❌ Git Authentication Failed (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Permission denied (publickey)",
  "error": "Git authentication failed. Ensure SSH key is added to GitHub as Deploy Key with write access"
}
```

**❌ BotAJS API Error (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Failed to fetch bot: 401 Unauthorized",
  "error": "Failed to fetch bot: 401 Unauthorized"
}
```

#### Repository Structure

The export creates the following structure in your Git repository:

```
repository-root/
├── bot.json                 # Bot metadata (botId, platform, username)
├── README.md               # Auto-generated documentation
└── commands/               # Bot commands directory
    ├── help.js            # Command: help
    ├── start.js           # Command: start
    └── status.js          # Command: status
```

#### Notes

- Requires SSH key to be generated first (`/api/git/generate-key`)
- SSH public key must be added to GitHub repository as Deploy Key
- Deploy Key must have **write access** enabled
- Clones repository, updates files, commits, and pushes
- Creates comprehensive README.md automatically
- Commands are base64-decoded and saved as `.js` files
- If branch doesn't exist, it will be created

---

### 5. Import Bot from GitHub

Import bot commands from a GitHub repository to BotAJS.

**Endpoint:** `POST /api/bots/import`

#### Request

```bash
curl -X POST http://your-domain.com/api/bots/import \
  -H "X-API-KEY: your-secret-api-key-here" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "user_123",
    "botId": "bot_abc123",
    "git_repo_url": "git@github.com:username/my-bot-repo.git",
    "git_branch": "main",
    "api_key": "botajs_api_key_here"
  }'
```

**Parameters:**

| Parameter      | Type   | Required | Default     | Description                                    |
|----------------|--------|----------|-------------|------------------------------------------------|
| user_id        | string | No       | "anonymous" | User identifier (must have SSH key configured) |
| botId          | string | **Yes**  | -           | BotAJS bot identifier                          |
| git_repo_url   | string | **Yes**  | -           | Git repository SSH URL                         |
| git_branch     | string | No       | "main"      | Source branch to import from                   |
| api_key        | string | **Yes**  | -           | BotAJS API key                                 |

#### Responses

**✅ Success (200 OK)**
```json
{
  "success": true,
  "message": "Bot imported successfully from GitHub",
  "data": {
    "message": "Bot imported successfully from GitHub",
    "bot_id": "bot_abc123",
    "repository": "git@github.com:username/my-bot-repo.git",
    "branch": "main",
    "commands_imported": 3,
    "commands": [
      {
        "command": "help",
        "status": "created"
      },
      {
        "command": "start",
        "status": "created"
      },
      {
        "command": "status",
        "status": "updated"
      }
    ]
  }
}
```

**❌ Validation Error (400 Bad Request)**
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": {
    "botId": "The botId field is required",
    "git_repo_url": "The git_repo_url field is required",
    "api_key": "The api_key field is required"
  }
}
```

**❌ SSH Key Not Configured (400 Bad Request)**
```json
{
  "success": false,
  "message": "SSH key not configured",
  "error": "SSH key not configured"
}
```

**❌ No Commands Found (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "No commands found in repository",
  "error": "No commands found in /commands directory"
}
```

**❌ Command Creation Failed (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Failed to create command: 409 New command already exists",
  "error": "Failed to create command: 409 New command already exists"
}
```

**❌ Git Clone Failed (500 Internal Server Error)**
```json
{
  "success": false,
  "message": "Permission denied (publickey)",
  "error": "Git authentication failed. Ensure SSH key is added to GitHub as Deploy Key"
}
```

#### Expected Repository Structure

The import expects this structure in your Git repository:

```
repository-root/
├── commands/               # Required: Commands directory
│   ├── help.js            # Command files (.js)
│   ├── start.js
│   └── status.js
└── bot.json               # Optional: Bot metadata
```

#### Command File Format

Each command file should contain JavaScript code:

```javascript
// commands/help.js
return "Here are available commands:\n" +
       "/start - Start the bot\n" +
       "/help - Show this help\n" +
       "/status - Check bot status";
```

#### Import Behavior

- **New commands**: Created in BotAJS
- **Existing commands**: Updated with new code (uses `old_command` parameter)
- **need_reply**: Defaults to `false` (0)
- **Code encoding**: Files are base64-encoded before upload to BotAJS

#### Notes

- Requires SSH key to be generated first
- SSH public key must be added to GitHub repository as Deploy Key
- Read-only Deploy Key is sufficient (no write access needed)
- Commands must be in `/commands` directory at repository root
- Only `.js` files are imported
- Handles both creating new commands and updating existing ones
- Files with empty or null content are skipped

---

## Error Codes

### Standard HTTP Status Codes

| Code | Status                | Description                                          |
|------|-----------------------|------------------------------------------------------|
| 200  | OK                    | Request successful                                   |
| 400  | Bad Request           | Validation failed or required parameters missing     |
| 401  | Unauthorized          | Invalid or missing API key                           |
| 404  | Not Found             | Resource not found (endpoint or SSH key)             |
| 500  | Internal Server Error | Server-side error (database, Git, BotAJS API, etc.)  |

### Error Response Format

All errors follow this format:

```json
{
  "success": false,
  "message": "Human-readable error message",
  "error": "Detailed error description",
  "errors": {
    "field_name": "Field-specific error message"
  }
}
```

### Common Error Messages

#### Authentication Errors
```json
{
  "success": false,
  "message": "Unauthorized",
  "error": "Invalid or missing API key"
}
```

#### Validation Errors
```json
{
  "success": false,
  "message": "Validation failed",
  "errors": {
    "botId": "The botId field is required",
    "git_repo_url": "The git_repo_url field is required"
  }
}
```

#### Database Errors
```json
{
  "success": false,
  "message": "Database connection failed: Access denied",
  "error": "Database connection failed: Access denied for user 'dannvtuc_botajs'@'localhost'"
}
```

#### Git Errors
```json
{
  "success": false,
  "message": "Permission denied (publickey)",
  "error": "Git authentication failed"
}
```

#### BotAJS API Errors
```json
{
  "success": false,
  "message": "Failed to fetch bot: 401 Unauthorized",
  "error": "Failed to fetch bot: 401 Unauthorized"
}
```

---

## Rate Limiting

Currently, there are no rate limits enforced by the API. However, consider:

- **GitHub API**: Subject to GitHub's API rate limits
- **BotAJS API**: Subject to BotAJS API rate limits
- **Git Operations**: Clone/push operations can be slow for large repositories

**Recommendations:**
- Implement client-side throttling for export/import operations
- Cache SSH key information locally
- Use webhooks instead of polling for repository changes

---

## Examples

### Complete Workflow Example

#### 1. Generate SSH Key

```bash
curl -X POST http://localhost:8000/api/git/generate-key \
  -H "X-API-KEY: my-secret-key" \
  -H "Content-Type: application/json" \
  -d '{"user_id": "john_123"}'
```

**Response:**
```json
{
  "success": true,
  "message": "Deploy key generated successfully",
  "data": {
    "public_key": "ssh-rsa AAAAB3NzaC1yc2EA..."
  }
}
```

#### 2. Add Key to GitHub

1. Copy the `public_key` from response
2. Go to: GitHub → Repository → Settings → Deploy keys
3. Click "Add deploy key"
4. Paste the public key
5. ✓ Check "Allow write access" (for export)
6. Save

#### 3. Check Key Status

```bash
curl -X POST http://localhost:8000/api/user/ssh-keys \
  -H "X-API-KEY: my-secret-key" \
  -H "Content-Type: application/json" \
  -d '{"user_id": "john_123"}'
```

**Response:**
```json
{
  "has_key": true,
  "message": "SSH key found",
  "public_key": "ssh-rsa AAAAB3NzaC1yc2EA...",
  "created_at": "2025-11-21 10:30:00",
  "key_fingerprint": "2f:3a:4b:5c:6d:7e:8f:9a"
}
```

#### 4. Export Bot to GitHub

```bash
curl -X POST http://localhost:8000/api/bots/export \
  -H "X-API-KEY: my-secret-key" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "john_123",
    "botId": "my_telegram_bot",
    "git_repo_url": "git@github.com:john/my-bot.git",
    "git_branch": "main",
    "commit_message": "Initial bot export",
    "user_email": "john@example.com",
    "api_key": "botajs_abc123"
  }'
```

**Response:**
```json
{
  "success": true,
  "message": "Bot exported successfully to GitHub",
  "data": {
    "bot_id": "my_telegram_bot",
    "repository": "git@github.com:john/my-bot.git",
    "branch": "main",
    "commands_exported": 5
  }
}
```

#### 5. Import Bot from GitHub

```bash
curl -X POST http://localhost:8000/api/bots/import \
  -H "X-API-KEY: my-secret-key" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": "john_123",
    "botId": "my_telegram_bot",
    "git_repo_url": "git@github.com:john/my-bot.git",
    "git_branch": "main",
    "api_key": "botajs_abc123"
  }'
```

**Response:**
```json
{
  "success": true,
  "message": "Bot imported successfully from GitHub",
  "data": {
    "commands_imported": 5,
    "commands": [
      {"command": "start", "status": "created"},
      {"command": "help", "status": "updated"}
    ]
  }
}
```

### JavaScript Integration Example

```javascript
class BotAJSGitAPI {
  constructor(apiKey, baseUrl) {
    this.apiKey = apiKey;
    this.baseUrl = baseUrl;
  }

  async request(endpoint, data) {
    const response = await fetch(`${this.baseUrl}${endpoint}`, {
      method: 'POST',
      headers: {
        'X-API-KEY': this.apiKey,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(data)
    });
    return response.json();
  }

  async generateSSHKey(userId) {
    return this.request('/api/git/generate-key', { user_id: userId });
  }

  async getUserSSHKeys(userId) {
    return this.request('/api/user/ssh-keys', { user_id: userId });
  }

  async exportBot(userId, botId, repoUrl, branch, commitMessage, email, apiKey) {
    return this.request('/api/bots/export', {
      user_id: userId,
      botId,
      git_repo_url: repoUrl,
      git_branch: branch,
      commit_message: commitMessage,
      user_email: email,
      api_key: apiKey
    });
  }

  async importBot(userId, botId, repoUrl, branch, apiKey) {
    return this.request('/api/bots/import', {
      user_id: userId,
      botId,
      git_repo_url: repoUrl,
      git_branch: branch,
      api_key: apiKey
    });
  }
}

// Usage
const api = new BotAJSGitAPI('my-secret-key', 'http://localhost:8000');

// Check if user has SSH key
const keyStatus = await api.getUserSSHKeys('john_123');
if (!keyStatus.has_key) {
  // Generate new key
  const result = await api.generateSSHKey('john_123');
  console.log('Add this to GitHub:', result.data.public_key);
}

// Export bot
const exportResult = await api.exportBot(
  'john_123',
  'my_telegram_bot',
  'git@github.com:john/my-bot.git',
  'main',
  'Update commands',
  'john@example.com',
  'botajs_abc123'
);
console.log('Export result:', exportResult);
```

### PHP Integration Example

```php
<?php

class BotAJSGitClient
{
    private $apiKey;
    private $baseUrl;

    public function __construct($apiKey, $baseUrl)
    {
        $this->apiKey = $apiKey;
        $this->baseUrl = $baseUrl;
    }

    private function request($endpoint, $data)
    {
        $ch = curl_init($this->baseUrl . $endpoint);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'X-API-KEY: ' . $this->apiKey,
            'Content-Type: application/json'
        ]);

        $response = curl_exec($ch);
        $statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return json_decode($response, true);
    }

    public function generateSSHKey($userId)
    {
        return $this->request('/api/git/generate-key', ['user_id' => $userId]);
    }

    public function getUserSSHKeys($userId)
    {
        return $this->request('/api/user/ssh-keys', ['user_id' => $userId]);
    }

    public function exportBot($userId, $botId, $repoUrl, $branch, $message, $email, $apiKey)
    {
        return $this->request('/api/bots/export', [
            'user_id' => $userId,
            'botId' => $botId,
            'git_repo_url' => $repoUrl,
            'git_branch' => $branch,
            'commit_message' => $message,
            'user_email' => $email,
            'api_key' => $apiKey
        ]);
    }

    public function importBot($userId, $botId, $repoUrl, $branch, $apiKey)
    {
        return $this->request('/api/bots/import', [
            'user_id' => $userId,
            'botId' => $botId,
            'git_repo_url' => $repoUrl,
            'git_branch' => $branch,
            'api_key' => $apiKey
        ]);
    }
}

// Usage
$client = new BotAJSGitClient('my-secret-key', 'http://localhost:8000');

// Check SSH key status
$status = $client->getUserSSHKeys('john_123');
if (!$status['has_key']) {
    $result = $client->generateSSHKey('john_123');
    echo "Add to GitHub: " . $result['data']['public_key'];
}

// Export bot
$result = $client->exportBot(
    'john_123',
    'my_telegram_bot',
    'git@github.com:john/my-bot.git',
    'main',
    'Update commands',
    'john@example.com',
    'botajs_abc123'
);
print_r($result);
```

---

## Testing

### Test with cURL

```bash
# Set variables
API_KEY="your-secret-api-key-here"
BASE_URL="http://localhost:8000"
USER_ID="test_user"
BOT_ID="your_bot_id"
BOTAJS_API_KEY="your_botajs_api_key"
REPO_URL="git@github.com:username/repo.git"

# 1. Generate SSH key
curl -X POST "$BASE_URL/api/git/generate-key" \
  -H "X-API-KEY: $API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"user_id\": \"$USER_ID\"}"

# 2. Get SSH key status
curl -X POST "$BASE_URL/api/user/ssh-keys" \
  -H "X-API-KEY: $API_KEY" \
  -H "Content-Type: application/json" \
  -d "{\"user_id\": \"$USER_ID\"}"

# 3. Export bot
curl -X POST "$BASE_URL/api/bots/export" \
  -H "X-API-KEY: $API_KEY" \
  -H "Content-Type: application/json" \
  -d "{
    \"user_id\": \"$USER_ID\",
    \"botId\": \"$BOT_ID\",
    \"git_repo_url\": \"$REPO_URL\",
    \"git_branch\": \"main\",
    \"commit_message\": \"Test export\",
    \"user_email\": \"test@example.com\",
    \"api_key\": \"$BOTAJS_API_KEY\"
  }"

# 4. Import bot
curl -X POST "$BASE_URL/api/bots/import" \
  -H "X-API-KEY: $API_KEY" \
  -H "Content-Type: application/json" \
  -d "{
    \"user_id\": \"$USER_ID\",
    \"botId\": \"$BOT_ID\",
    \"git_repo_url\": \"$REPO_URL\",
    \"git_branch\": \"main\",
    \"api_key\": \"$BOTAJS_API_KEY\"
  }"
```

### Test with Postman

1. Import the collection from `postman_collection.json` (if provided)
2. Set environment variables:
   - `api_key`: Your API key
   - `base_url`: API base URL
   - `user_id`: Test user ID
   - `bot_id`: Your bot ID
   - `botajs_api_key`: Your BotAJS API key
   - `repo_url`: Your GitHub repository URL
3. Run requests in sequence

---

## Support

For issues and questions:

- **Documentation**: [README.md](README.md)
- **Database Setup**: [DATABASE_SETUP.md](DATABASE_SETUP.md)
- **Changes Log**: [CHANGES.md](CHANGES.md)
- **Quick Start**: [QUICK_START.md](QUICK_START.md)

---

**API Version:** 2.0.0
**Last Updated:** 2025-11-21
**Base URL:** `http://your-domain.com`
